// ==UserScript==
// @name         5ch とりあえずNG
// @namespace    http://tampermonkey.net/
// @version      2.1
// @description  日本語50文字以上 + [数字]パターン2回以上の投稿を非表示。他機能あり。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    if (window.top !== window.self) {
        return; // iframe内では実行しない
    }

    // === カスタマイズ ===
    const ID_BLACKLIST = ['abc123', 'xyz999']; // 非表示対象のIDを追加
    const SIMILARITY_THRESHOLD = 0.75; // 類似度スコアの閾値（0〜1）

    let hiddenCount = 0;
    let hiddenPosts = [];

    function createControlPanel() {
        const panel = document.createElement('div');
        panel.style.position = 'fixed';
        panel.style.bottom = '55px'; /* 下からの位置指定 */
        panel.style.left = '10px';
        panel.style.background = '#0e0e0e';
        panel.style.opacity = 0.3;
        panel.style.color = '#e0e0e0';
        panel.style.padding = '8px';
        panel.style.border = '0px solid #999';
        panel.style.zIndex = 9999;
        panel.style.fontSize = '13px';
        panel.style.borderRadius = '4px';
        panel.style.pointerEvents = 'auto';

        panel.innerHTML = `
        NG: <span id="ng-count">${hiddenCount}</span> 件
        <button id="toggle-hidden" style="
            margin-left: 8px;
            font-size: 13px;
            font-weight: normal;
            padding: 2px 6px;
            background: #0e0e0e;
            color: #eee;
            border: 1px solid #666;
            border-radius: 3px;
            cursor: pointer;
            display: inline-block;
            vertical-align: middle;
            line-height: 1;
            margin-bottom: 0;
        ">非表示</button>
    `;

        document.body.appendChild(panel);

        // 一時表示ボタンの切り替え機能
        document.getElementById('toggle-hidden').addEventListener('click', function () {
            // 1. 中央に最も近い投稿要素を取得
            const posts = Array.from(document.querySelectorAll('.post'))
            .filter(el => el.offsetParent !== null); // 表示されてるものだけ

            const centerY = window.innerHeight / 2;
            let closestPost = null;
            let minDistance = Infinity;

            posts.forEach(post => {
                const rect = post.getBoundingClientRect();
                const distance = Math.abs(rect.top - centerY);
                if (distance < minDistance) {
                    minDistance = distance;
                    closestPost = post;
                }
            });

            const topBefore = closestPost?.getBoundingClientRect().top ?? 0;

            // 2. 表示・非表示を切り替える
            const isHidden = hiddenPosts[0]?.closest('.post')?.style.display === 'none';
            hiddenPosts.forEach(post => {
                const el = post.closest('.post');
                if (el) {
                    el.style.display = isHidden ? '' : 'none';
                }
            });

            // 3. 同じ投稿の位置の差分だけスクロールを補正
            const topAfter = closestPost?.getBoundingClientRect().top ?? 0;
            const delta = topAfter - topBefore;
            window.scrollBy({ top: delta });

            // 4. ボタンラベル切り替え
            this.textContent = isHidden ? '表示中' : '非表示';
        });

    }

    function shouldHide(postContent) {
        const div = document.createElement('div');
        div.innerHTML = postContent.innerHTML;
        const text = div.textContent;

        // 条件①：日本語50文字以上
        const japaneseChars = text.match(/[\u3040-\u30FF\u4E00-\u9FAF\uFF66-\uFF9D]/g);
        const isLongJapanese = japaneseChars && japaneseChars.length >= 50;

        // 条件②：[1234567]のようなパターンが2回以上
        const brackets = text.match(/\[\d{6,9}\]/g);
        const hasTwoBrackets = brackets && brackets.length >= 2;

        // 条件③：URLリンクを含まない
        const hasNoLinks = !text.match(/https?:\/\/\S+/);

        // 条件①＋②＋③がすべて成立した場合のみ true
        if (isLongJapanese && hasTwoBrackets && hasNoLinks) return true;

        // 条件④：IDブラックリスト
        const idElem = postContent.closest('.post')?.querySelector('.postuid');
        if (idElem) {
            const idMatch = idElem.textContent.match(/ID:([A-Za-z0-9+/]+)/);
            if (idMatch && ID_BLACKLIST.includes(idMatch[1])) return true;
        }

        // 条件⑤：類似投稿と高スコア一致
        const simScore = getSimilarityScore(text);
        if (simScore >= SIMILARITY_THRESHOLD) return true;

        return false;
    }

    function getSimilarityScore(currentText) {
        let maxScore = 0;
        hiddenPosts.forEach(prev => {
            const prevText = prev.innerText;
            const score = simpleSimilarity(currentText, prevText);
            if (score > maxScore) maxScore = score;
        });
        return maxScore;
    }

    function simpleSimilarity(text1, text2) {
        const a = new Set(text1.split(/\s+/));
        const b = new Set(text2.split(/\s+/));
        const intersection = new Set([...a].filter(x => b.has(x)));
        const union = new Set([...a, ...b]);
        return intersection.size / union.size;
    }

    function hideMatchingPosts() {
        const container = document.querySelector('#threadcontent');
        if (!container) return;

        const posts = container.querySelectorAll('.post-content');
        posts.forEach(post => {
            if (!post.dataset.checked) {
                if (shouldHide(post)) {
                    const postWrapper = post.closest('.post');
                    if (postWrapper) {
                        postWrapper.style.display = 'none';
                        hiddenPosts.push(post);
                        hiddenCount++;
                        const countElem = document.getElementById('ng-count');
                        if (countElem) countElem.textContent = hiddenCount;
                    }
                }
                post.dataset.checked = 'true';
            }
        });
    }

    // 初期処理
    hideMatchingPosts();
    createControlPanel();

    // 動的変化の監視
    const container = document.querySelector('#threadcontent');
    if (container) {
        let timeout;
        const observer = new MutationObserver(() => {
            clearTimeout(timeout);
            timeout = setTimeout(() => hideMatchingPosts(), 300);
        });
        observer.observe(container, { childList: true, subtree: true });
    }
})();
